<?php
/**
 * Treweler Core Functions
 * General core functions available on both the front-end and admin.
 *
 * @package Treweler\Functions
 * @version 0.23
 */

if ( ! defined( 'ABSPATH' ) ) {
  exit;
}

/**
 * Wrapper for _doing_it_wrong().
 *
 * @param string $function Function used.
 * @param string $message Message to log.
 * @param string $version Version the message was added in.
 *
 * @since  0.23
 */
function twer_doing_it_wrong( $function, $message, $version ) {
  // @codingStandardsIgnoreStart
  $message .= esc_html__( ' Backtrace: ', 'treweler' ) . wp_debug_backtrace_summary();

  if ( is_ajax() || TWER()->is_rest_api_request() ) {
    do_action( 'doing_it_wrong_run', $function, $message, $version );
    error_log( esc_html__( "{$function} was called incorrectly. {$message}. This message was added in version {$version}." ) );
  } else {
    _doing_it_wrong( $function, $message, $version );
  }
  // @codingStandardsIgnoreEnd
}


if ( ! function_exists( 'is_ajax' ) ) {

  /**
   * Is_ajax - Returns true when the page is loaded via ajax.
   *
   * @return bool
   */
  function is_ajax() {
    return function_exists( 'wp_doing_ajax' ) ? wp_doing_ajax() : defined( 'DOING_AJAX' );
  }
}

if ( ! function_exists( 'twer_clean' ) ) {
  /**
   * Clean variables using sanitize_text_field. Arrays are cleaned recursively.
   * Non-scalar values are ignored.
   *
   * @param string|array $var Data to sanitize.
   *
   * @return string|array
   */
  function twer_clean( $var ) {
    if ( is_array( $var ) ) {
      return array_map( 'twer_clean', $var );
    } else {
      return is_scalar( $var ) ? sanitize_text_field( $var ) : $var;
    }
  }
}

if ( ! function_exists( 'twer_get_data' ) ) {
  /**
   * Parse all meta data and store it
   *
   * @param $post_id
   * @param string $prefix
   *
   * @return object
   */
  function twer_get_data( $post_id = null, $prefix = '_treweler_' ) {
    global $post;

    if ( empty( $post_id ) ) {
      $post_id = isset( $post->ID ) ? $post->ID : 0;
    }

    $post_meta = get_post_meta( $post_id );
    $meta_data = [];

    if ( 'publish' !== get_post_status( $post_id ) ) {
      return (object) $meta_data;
    }

    if ( $post_meta ) {
      foreach ( $post_meta as $meta_key => $meta_value ) {
        if ( strpos( $meta_key, $prefix ) !== false ) {

          if ( is_array( $meta_value ) && count( $meta_value ) === 1 ) {
            $meta_value = maybe_unserialize( reset( $meta_value ) );
          }
          $meta_data[ str_replace( $prefix, '', $meta_key ) ] = $meta_value;
        }
      }
    }

    return (object) $meta_data;
  }
}

/**
 * Define a constant if it is not already defined.
 *
 * @param string $name Constant name.
 * @param mixed $value Value.
 *
 * @since 1.03
 */
function twer_maybe_define_constant( $name, $value ) {
  if ( ! defined( $name ) ) {
    define( $name, $value );
  }
}

/**
 * Flushes rewrite rules when the map page (or it's children) gets saved.
 */
function flush_rewrite_rules_on_map_page_save() {
  $screen = get_current_screen();
  $screen_id = $screen ? $screen->id : '';

  // Check if this is the edit page.
  if ( 'page' !== $screen_id ) {
    return;
  }

  // Check if page is edited.
  if ( empty( $_GET['post'] ) || empty( $_GET['action'] ) || ( isset( $_GET['action'] ) && 'edit' !== $_GET['action'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    return;
  }

  $post_id = intval( $_GET['post'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
  $setMap = get_post_meta( $post_id, '_treweler_cpt_dd_box_fullscreen', true );

  if ( ! empty( $setMap ) ) {
    do_action( 'treweler_flush_rewrite_rules' );
  }
}

add_action( 'admin_footer', 'flush_rewrite_rules_on_map_page_save' );


if ( ! function_exists( 'twer_get_meta' ) ) {
  /**
   * Get meta data
   *
   * @param string $meta_name
   * @param string|int $post_id
   * @param string $prefix
   *
   * @return mixed
   */
  function twer_get_meta( string $meta_name, int $post_id = 0, string $prefix = '_treweler_' ) {
    return get_post_meta( $post_id, $prefix . $meta_name, true );
  }
}


if ( ! function_exists( 'twer_get_api_key' ) ) {
  /**
   * Get Mapbox API key
   *
   * @return mixed
   */
  function twer_get_api_key() {
    $option = get_option( 'treweler' );

    return isset( $option['api_key'] ) ? $option['api_key'] : '';
  }
}

if ( ! function_exists( 'twer_is_valid_apikey' ) ) {
  /**
   * Check if current API key is valid
   *
   * @return bool
   */
  function twer_is_valid_apikey() {
    $option = get_option( 'treweler' );

    return empty( $option['api_key_invalid'] );
  }
}

if ( ! function_exists( 'twer_minify_html_markup' ) ) {
  /**
   * Minify HTML markup
   *
   * @param $text
   *
   * @return string|string[]|null
   */
  function twer_minify_html_markup( $text ) {
    $search = array(
      '/\>[^\S ]+/s',     // strip whitespaces after tags, except space
      '/[^\S ]+\</s',     // strip whitespaces before tags, except space
      '/(\s)+/s',         // shorten multiple whitespace sequences
      '/<!--(.|\s)*?-->/' // Remove HTML comments
    );

    $replace = array(
      '>',
      '<',
      '\\1',
      ''
    );

    return preg_replace( $search, $replace, $text );
  }
}

if ( ! function_exists( 'twer_fix_html_markup' ) ) {
  /**
   * Fix HTML markup
   *
   * @param $text
   *
   * @return string|string[]
   */
  function twer_fix_html_markup( $text ) {
    $tagstack = array();
    $stacksize = 0;
    $tagqueue = '';
    $newtext = '';
    // Known single-entity/self-closing tags.
    $single_tags = array(
      'area',
      'base',
      'basefont',
      'br',
      'col',
      'command',
      'embed',
      'frame',
      'hr',
      'img',
      'input',
      'isindex',
      'link',
      'meta',
      'param',
      'source'
    );
    // Tags that can be immediately nested within themselves.
    $nestable_tags = array( 'blockquote', 'div', 'object', 'q', 'span' );

    // WP bug fix for comments - in case you REALLY meant to type '< !--'.
    $text = str_replace( '< !--', '<    !--', $text );
    // WP bug fix for LOVE <3 (and other situations with '<' before a number).
    $text = preg_replace( '#<([0-9]{1})#', '&lt;$1', $text );

    /**
     * Matches supported tags.
     * To get the pattern as a string without the comments paste into a PHP
     * REPL like `php -a`.
     *
     * @see https://html.spec.whatwg.org/#elements-2
     * @see https://w3c.github.io/webcomponents/spec/custom/#valid-custom-element-name
     * @example
     * ~# php -a
     * php > $s = [paste copied contents of expression below including parentheses];
     */
    $tag_pattern = (
      '#<' . // Start with an opening bracket.
      '(/?)' . // Group 1 - If it's a closing tag it'll have a leading slash.
      '(' . // Group 2 - Tag name.
      // Custom element tags have more lenient rules than HTML tag names.
      '(?:[a-z](?:[a-z0-9._]*)-(?:[a-z0-9._-]+)+)' .
      '|' .
      // Traditional tag rules approximate HTML tag names.
      '(?:[\w:]+)' .
      ')' .
      '(?:' .
      // We either immediately close the tag with its '>' and have nothing here.
      '\s*' .
      '(/?)' . // Group 3 - "attributes" for empty tag.
      '|' .
      // Or we must start with space characters to separate the tag name from the attributes (or whitespace).
      '(\s+)' . // Group 4 - Pre-attribute whitespace.
      '([^>]*)' . // Group 5 - Attributes.
      ')' .
      '>#' // End with a closing bracket.
    );

    while ( preg_match( $tag_pattern, $text, $regex ) ) {
      $full_match = $regex[0];
      $has_leading_slash = ! empty( $regex[1] );
      $tag_name = $regex[2];
      $tag = strtolower( $tag_name );
      $is_single_tag = in_array( $tag, $single_tags, true );
      $pre_attribute_ws = isset( $regex[4] ) ? $regex[4] : '';
      $attributes = trim( isset( $regex[5] ) ? $regex[5] : $regex[3] );
      $has_self_closer = '/' === substr( $attributes, - 1 );

      $newtext .= $tagqueue;

      $i = strpos( $text, $full_match );
      $l = strlen( $full_match );

      // Clear the shifter.
      $tagqueue = '';
      if ( $has_leading_slash ) { // End tag.
        // If too many closing tags.
        if ( $stacksize <= 0 ) {
          $tag = '';
          // Or close to be safe $tag = '/' . $tag.

          // If stacktop value = tag close value, then pop.
        } elseif ( $tagstack[ $stacksize - 1 ] === $tag ) { // Found closing tag.
          $tag = '</' . $tag . '>'; // Close tag.
          array_pop( $tagstack );
          $stacksize --;
        } else { // Closing tag not at top, search for it.
          for ( $j = $stacksize - 1; $j >= 0; $j -- ) {
            if ( $tagstack[ $j ] === $tag ) {
              // Add tag to tagqueue.
              for ( $k = $stacksize - 1; $k >= $j; $k -- ) {
                $tagqueue .= '</' . array_pop( $tagstack ) . '>';
                $stacksize --;
              }
              break;
            }
          }
          $tag = '';
        }
      } else { // Begin tag.
        if ( $has_self_closer ) { // If it presents itself as a self-closing tag...
          // ...but it isn't a known single-entity self-closing tag, then don't let it be treated as such
          // and immediately close it with a closing tag (the tag will encapsulate no text as a result).
          if ( ! $is_single_tag ) {
            $attributes = trim( substr( $attributes, 0, - 1 ) ) . "></$tag";
          }
        } elseif ( $is_single_tag ) { // Else if it's a known single-entity tag but it doesn't close itself, do so.
          $pre_attribute_ws = ' ';
          $attributes .= '/';
        } else { // It's not a single-entity tag.
          // If the top of the stack is the same as the tag we want to push, close previous tag.
          if ( $stacksize > 0 && ! in_array( $tag, $nestable_tags,
              true ) && $tagstack[ $stacksize - 1 ] === $tag ) {
            $tagqueue = '</' . array_pop( $tagstack ) . '>';
            $stacksize --;
          }
          $stacksize = array_push( $tagstack, $tag );
        }

        // Attributes.
        if ( $has_self_closer && $is_single_tag ) {
          // We need some space - avoid <br/> and prefer <br />.
          $pre_attribute_ws = ' ';
        }

        $tag = '<' . $tag . $pre_attribute_ws . $attributes . '>';
        // If already queuing a close tag, then put this tag on too.
        if ( ! empty( $tagqueue ) ) {
          $tagqueue .= $tag;
          $tag = '';
        }
      }
      $newtext .= substr( $text, 0, $i ) . $tag;
      $text = substr( $text, $i + $l );
    }

    // Clear tag queue.
    $newtext .= $tagqueue;

    // Add remaining text.
    $newtext .= $text;

    while ( $x = array_pop( $tagstack ) ) {
      $newtext .= '</' . $x . '>'; // Add remaining tags to close.
    }

    // WP fix for the bug with HTML comments.
    $newtext = str_replace( '< !--', '<!--', $newtext );
    $newtext = str_replace( '<    !--', '< !--', $newtext );

    return $newtext;
  }
}


if ( ! function_exists( 'twer_is_map_in_page' ) ) {

  /**
   * Is_map_on_page - Returns true when viewing the full screen map page OR iframe map.
   *
   * @param string $id
   *
   * @return bool
   */
  function twer_is_map_in_page( $id = null ) {
    return twer_is_map_fullscreen( $id ) || twer_is_map_iframe();
  }
}

if ( ! function_exists( 'twer_is_map_fullscreen' ) ) {
  /**
   * Check if current map is fullscreen in show on single page
   *
   * @param null $id
   *
   * @return bool
   */
  function twer_is_map_fullscreen( $id = null ) {
    if ( ! $id ) {
      $id = get_the_ID();
    }
    $fs_meta = get_post_meta( $id, '_treweler_cpt_dd_box_fullscreen', true );

    return ( trim( $fs_meta ) !== '' && $fs_meta > 0 );
  }
}

if ( ! function_exists( 'twer_get_map_id' ) ) {
  /**
   * Get showed current Map ID
   *
   * @return int
   */
  function twer_get_map_id() {
    $map_id = 0;
    if ( twer_is_map_iframe() ) {
      $map_id = twer_get_iframe_map_id();
    } else {
      $map_id = get_post_meta( get_the_ID(), '_treweler_cpt_dd_box_fullscreen', true );
    }

    return (int) $map_id;
  }
}

if ( ! function_exists( 'twer_is_map_iframe' ) ) {
  /**
   * Check if current map show via ONLY iframe
   *
   * @return bool
   */
  function twer_is_map_iframe() {
    $params = twer_get_iframe_params();
    $is_iframe = ! empty( $params['tw'] ) ? $params['tw'] : '';

    return $is_iframe === 'iframe';
  }
}


if ( ! function_exists( 'twer_get_iframe_params' ) ) {
  /**
   * Get all shortcode iframe params via GET request
   *
   * @return array
   */
  function twer_get_iframe_params() {
    $params = [];
    if ( ! empty( $_GET ) ) {
      foreach ( $_GET as $key => $value ) {
        if ( strpos( $key, 'tw' ) !== false ) {
          $params[ str_replace( 'twer-', '', $key ) ] = $value;
        }
      }
    }

    return $params;
  }
}

if ( ! function_exists( 'twer_get_iframe_map_id' ) ) {
  /**
   * Get map id from iframe shortcode
   *
   * @return mixed|string
   */
  function twer_get_iframe_map_id() {
    $params = twer_get_iframe_params();

    return ! empty( $params['map-id'] ) ? $params['map-id'] : '';
  }
}


if ( ! function_exists( 'twer_load_template' ) ) {
  /**
   * Loads template from `treweler` child/main theme folder, if not exist from - PLUGIN_NAME/includes/ path
   *
   * @param null $file
   *
   * @return string
   */
  function twer_load_template( $file = null ) {
    $file_path = get_theme_file_path( 'treweler/' . $file );

    return file_exists( $file_path ) ? $file_path : untrailingslashit( plugin_dir_path( TWER_PLUGIN_FILE ) ) . '/includes/' . $file;
  }
}

if ( ! function_exists( 'twer_get_categories' ) ) {
  /**
   * Get Category List by Post ID
   *
   * @return mixed
   */
  function twer_get_categories( $post_id, $encode = true ) {
    $mask_results = [];
    $terms = get_the_terms( $post_id, 'map-category' );
    $i = 0;

    if ( isset( $terms ) && is_array( $terms ) ) {
      foreach ( $terms as $t ) {
        $mask_results[ $i ]['id'] = $t->term_id;
        $mask_results[ $i ]['name'] = $t->name;
        $mask_results[ $i ]['slug'] = $t->slug;
        $i ++;
      }
    } else {
      return 0;
    }


    if ( $encode ) {
      return htmlspecialchars( json_encode( $mask_results ), ENT_QUOTES, 'UTF-8' );
    }

    return $mask_results;

  }
}

if ( ! function_exists( 'twer_build_category_classes' ) ) {
  /**
   * Build Category Classes
   *
   * @return mixed
   */
  function twer_build_category_classes( $post_id, $selector = 'id', $separator = ' ' ) {

    if ( $post_id === 0 ) {
      return implode( $separator, [ 'twer-no-category' ] );
    }

    $cat = twer_get_categories( $post_id, false );

    if ( $cat >= 1 ) {
      $ids = [ 'twer-has-category' ];
      foreach ( $cat as $c ) {
        $ids[] = 'category-' . $c[ $selector ];
      }
    } else {
      $ids = [ 'twer-no-category' ];
    }


    return implode( $separator, $ids );
  }
}

if ( ! function_exists( 'twer_get_categories_ids' ) ) {
  /**
   * Get categories id's
   *
   * @return mixed
   */
  function twer_get_categories_ids( $post_id, $selector = 'id' ) {

    $ids = [];
    if ( $post_id === 0 ) {
      return '';
    }

    $cat = twer_get_categories( $post_id, false );

    if ( $cat >= 1 ) {
      foreach ( $cat as $c ) {
        $ids[] = $c[ $selector ];
      }
    }


    return wp_json_encode( $ids );
  }
}

if ( ! function_exists( 'wp_dropdown_cpt' ) ) {
  function wp_dropdown_cpt( $args = '' ) {

    $defaults = array(
      'show_option_all'       => esc_html__( "Show All Maps", 'treweler' ),
      'numberposts'           => - 1,
      'category'              => 0,
      'orderby'               => 'date',
      'order'                 => 'DESC',
      'include'               => array(),
      'exclude'               => array(),
      'meta_key'              => '',
      'meta_value'            => '',
      'post_type'             => 'map',
      'suppress_filters'      => true,
      'depth'                 => 0,
      'child_of'              => 0,
      'selected'              => 0,
      'echo'                  => 1,
      'name_filter'           => 'map_filter',
      'id'                    => '',
      'class'                 => '',
      'show_option_none'      => '',
      'show_option_no_change' => '',
      'option_none_value'     => '',
      'value_field'           => 'ID',
    );

    $parsed_args = wp_parse_args( $args, $defaults );

    $cpts = get_posts( array(
      'category'         => 0,
      'orderby'          => 'date',
      'order'            => 'DESC',
      'post__in'         => $parsed_args['include'],
      'exclude'          => array(),
      'post_type'        => $parsed_args['post_type'],
      'post_status'      => 'publish',
      'meta_key'         => '',
      'meta_value'       => '',
      'posts_per_page'   => $parsed_args['numberposts'],
      'suppress_filters' => $parsed_args['suppress_filters'],
    ) );

    // bypass if post__in parameter exist
    $cache_name = sanitize_title( 'twer_cpt_' . $parsed_args['post_type'] . '__list' );
    if ( ! isset( $parsed_args['include'] ) || isset( $_GET['post_type'] ) || isset( $_GET['map_filter'] ) ) {
      if ( isset( $cpts ) && is_array( $cpts ) ) {
        if ( count( $cpts ) >= 1 ) {
          set_transient( $cache_name, $cpts, 3500 );
        }
      }
      $cpts = get_transient( $cache_name );
    }

    $output = '';
    // Back-compat with old system where both id and name were based on $name argument.
    if ( empty( $parsed_args['id'] ) ) {
      $parsed_args['id'] = $parsed_args['name_filter'];
    }

    //if ( ! empty( $cpts ) ) {
    $class = '';
    if ( ! empty( $parsed_args['class'] ) ) {
      $class = " class='" . esc_attr( $parsed_args['class'] ) . "'";
    }
    $selected = ( '0' === (string) $parsed_args['selected'] ) ? " selected='selected'" : '';

    $output = "<select name='" . esc_attr( $parsed_args['name_filter'] ) . "'" . $class . " id='" . esc_attr( $parsed_args['id'] ) . "'>\n";
    if ( $parsed_args['show_option_all'] ) {
      $output .= "\t<option value='0'$selected>" . $parsed_args['show_option_all'] . "</option>\n";
    }

    if ( $parsed_args['show_option_no_change'] ) {
      $output .= "\t<option value=\"-1\"$selected>" . $parsed_args['show_option_no_change'] . "</option>\n";
    }
    if ( $parsed_args['show_option_none'] ) {
      $output .= "\t<option value=\"" . esc_attr( $parsed_args['option_none_value'] ) . '$selected">' . $parsed_args['show_option_none'] . "</option>\n";
    }
    $output .= walk_page_dropdown_tree( $cpts, $parsed_args['depth'], $parsed_args );
    $output .= "</select>\n";
    //}

    $html = $output;

    if ( $parsed_args['echo'] ) {
      echo $html;
    }

    return $html;
  }
}


/**
 * Isset and set default value if data empty or not valid
 *
 * @var $type string | array  | int | boolean
 * @var $return string boolean |
 */
if ( ! function_exists( 'twer_isset' ) ) {
  function twer_isset( $var = "", $default = false, $return = 'boolean', $type = 'array' ) {
    if ( $type === 'array' && is_array( $var ) ) {
      if ( isset( $var[0] ) ) {
        if ( $return === 'boolean' ) {
          return true;
        } else {
          return $var;
        }

      } else {
        return false;
      }
    } else {
      return $default;
    }

  }
}
/**
 * Internal Debugging Tools
 */

if ( ! function_exists( 'twer_write_log' ) ) {

  function twer_write_log( $log ) {
    if ( true === WP_DEBUG ) {
      if ( is_array( $log ) || is_object( $log ) ) {
        error_log( print_r( $log, true ) );
      } else {
        error_log( $log );
      }
    }
  }

}

if ( ! function_exists( 'twer_untrash_post_status' ) ) {
  /**
   * Remove trash status for some CPT
   *
   * @param $new_status
   * @param $post_id
   * @param $previous_status
   *
   * @return mixed|string
   */
  function twer_untrash_post_status( $new_status, $post_id, $previous_status ) {
    $post_type = get_post_type( $post_id );
    if ( $post_type === 'twer-custom-fields' || $post_type === 'twer-templates' ) {
      $new_status = 'publish';
    }

    return $new_status;
  }

  add_filter( 'wp_untrash_post_status', 'twer_untrash_post_status', 3, 10 );
}

if ( ! function_exists( 'twer_implode_r' ) ) {
  /**
   * Improved recursive function
   *
   * @param $glue
   * @param array $arr
   *
   * @return string
   */
  function twer_implode_r( $glue, array $arr ) {
    $ret = '';

    foreach ( $arr as $piece ) {
      if ( is_array( $piece ) ) {
        $ret .= $glue . twer_implode_r( $glue, $piece );
      } else {
        $ret .= $glue . $piece;
      }
    }

    return $ret;
  }
}

if ( ! function_exists( 'twer_get_marker_styles' ) ) {
  function twer_get_marker_styles( $post_id ) {
    $meta_data = twer_get_data( $post_id );
    $marker_size_default = get_post_meta( $post_id, '_treweler_marker_icon_size', true );
    $marker_styles = [];
    $need_fields = [
      'marker_style'           => 'default',
      'point_color'            => '#4b7715',
      'point_halo_color'       => '#ffffff',
      'point_halo_opacity'     => '0.5',
      'dot_icon_show'          => false,
      'dot_icon_picker'        => '',
      'dot_icon'               => [
        'color' => '#ffffff',
        'size'  => '15',
      ],
      'dot'                    => [
        'color' => '#4b7715',
        'size'  => '12',

      ],
      'dot_border'             => [
        'color' => '#ffffff',
        'size'  => '0',
      ],
      'dot_corner_radius'      => [
        'size'  => '50',
        'units' => '%'
      ],
      'balloon_icon_show'      => false,
      'balloon_icon_picker'    => '',
      'balloon_icon'           => [
        'color' => '#ffffff',
        'size'  => '15',
      ],
      'balloon'                => [
        'color' => '#4b7715',
        'size'  => '18',
      ],
      'balloon_border'         => [
        'color' => '#4b7715',
        'size'  => '4',
      ],
      'balloon_dot'            => [
        'color' => '#ffffff',
        'size'  => '8',
      ],
      'triangle_color'         => '#4b7715',
      'triangle_width'         => '12',
      'triangle_height'        => '10',
      'custom_marker_img'      => '',
      'custom_marker_position' => 'center',
      'custom_marker_size'     => '42',
      'custom_marker_cursor'   => 'pointer',
    ];


    foreach ( $need_fields as $key => $default_value ) {
      $value = isset( $meta_data->{$key} ) ? $meta_data->{$key} : '';

      if ( empty( $value ) && ! is_numeric( $value ) && ! is_bool( $value ) && ! is_array( $value ) ) {
        $value = $default_value;
      }

      if ( is_array( $value ) ) {
        foreach ( $value as $value_child_key => $value_child ) {
          if ( empty( $value_child ) && ! is_numeric( $value_child ) && ! is_bool( $value_child ) && ! is_array( $value_child ) ) {
            $default_value_child = isset( $default_value[ $value_child_key ] ) ? $default_value[ $value_child_key ] : '';
            $value[ $value_child_key ] = $default_value_child;
          }
        }
      }

      $marker_styles[ $key ] = $value;
    }


    if ( is_numeric( $marker_styles['custom_marker_img'] ) ) {
      $data_item = wp_get_attachment_image_src( $marker_styles['custom_marker_img'], 'full' );
      $marker_styles['custom_marker_img'] = isset( $data_item[0] ) ? $data_item[0] : $data_item;
    }

    $marker_styles['custom_marker_size_default'] = $marker_size_default;

    return $marker_styles;
  }
}

if ( ! function_exists( 'twer_map_has_shapes' ) ) {
  /**
   * Detect if map has any shape
   *
   * @param int $map_id
   *
   * @return bool
   */
  function twer_map_has_shapes( int $map_id ) {
    global $post;
    $map_has_shapes = false;

    $tmp_post = $post;
    $shapes = get_posts( [
      'posts_per_page' => - 1,
      'numberposts'    => - 1,
      'post_type'      => 'twer-shapes',
    ] );
    foreach ( $shapes as $post ) {
      setup_postdata( $post );
      $shape_id = get_the_ID();

      $maps_ids = get_post_meta( $shape_id, '_treweler_map_id', true );

      if(!empty($maps_ids) && is_array($maps_ids)) {
        $maps_ids = array_map( static function( $value ) {
          return (int) $value;
        }, $maps_ids );

        if ( in_array( $map_id, $maps_ids, true ) ) {
          $map_has_shapes = true;
          break;
        }
      }
    }

    $post = $tmp_post;

    return $map_has_shapes;
  }
}

if ( ! function_exists( 'twer_map_has_routes' ) ) {
  /**
   * Detect if map has any route
   *
   * @param int $map_id
   *
   * @return bool
   */
  function twer_map_has_routes( int $map_id ) {
    global $post;
    $map_has_routes = false;

    $tmp_post = $post;
    $routes = get_posts( [
      'posts_per_page' => - 1,
      'numberposts'    => - 1,
      'post_type'      => 'route',
    ] );
    foreach ( $routes as $post ) {
      setup_postdata( $post );
      $route_id = get_the_ID();

      $maps_ids = get_post_meta( $route_id, '_treweler_route_map_id', true );

      if(!empty($maps_ids)) {
        if(!is_array($maps_ids)) {
          $maps_ids = [$maps_ids];
        }

        $maps_ids = array_map( static function( $value ) {
          return (int) $value;
        }, $maps_ids );

        if ( in_array( $map_id, $maps_ids, true ) ) {
          $map_has_routes = true;
          break;
        }
      }
    }

    $post = $tmp_post;

    return $map_has_routes;
  }
}
