<?php
/**
 * Post Types Admin
 *
 * @package  Treweler/admin
 * @version  0.24
 */

if ( ! defined( 'ABSPATH' ) ) {
  exit;
}

if ( class_exists( 'TWER_Admin_Post_Types', false ) ) {
  new TWER_Admin_Post_Types();

  return;
}

/**
 * TWER_Admin_Post_Types Class.
 * Handles the edit posts views and some functionality on the edit post screen for TWER post types.
 */
class TWER_Admin_Post_Types {

  /**
   * Constructor.
   */
  public function __construct() {
    include_once dirname( __FILE__ ) . '/class-twer-admin-meta-boxes.php';

    // Load correct list table classes for current screen.
    add_action( 'current_screen', [ $this, 'setup_screen' ] );
    add_action( 'check_ajax_referer', [ $this, 'setup_screen' ] );

    // Extra post data and screen elements.
    add_action( 'edit_form_top', [ $this, 'edit_form_top' ] );
    add_action( 'edit_form_after_title', [ $this, 'edit_form_after_title' ] );
    add_filter( 'enter_title_here', [ $this, 'enter_title_here' ], 1, 2 );
    add_filter( 'default_hidden_meta_boxes', [ $this, 'hidden_meta_boxes' ], 10, 2 );

    // Admin notices.
    add_filter( 'post_updated_messages', [ $this, 'post_updated_messages' ] );
    add_filter( 'bulk_post_updated_messages', [ $this, 'bulk_post_updated_messages' ], 10, 2 );

    // Uploads.
    add_filter( 'upload_mimes', [ $this, 'add_new_mime_types' ] );
    add_filter( 'wp_check_filetype_and_ext', [ $this, 'add_allow_upload_extension_exception' ], 10, 5 );
  }

  /**
   * Add new mime types for media uploads
   *
   * @param $mime_types
   *
   * @return mixed
   */
  public function add_new_mime_types( $mime_types ) {
    $mime_types['svg'] = 'image/svg+xml';
    $mime_types['gpx'] = 'application/gpx+xml';

    return $mime_types;
  }

  /**
   * Exception for WordPress 4.7.1 file contents check system using finfo_file (wp-includes/functions.php)
   * In case of custom extension in this plugins' setting, the WordPress 4.7.1 file contents check system is always true.
   *
   * @param $data
   * @param $file
   * @param $filename
   * @param $mimes
   * @param $real_mime
   *
   * @return array
   */
  public function add_allow_upload_extension_exception( $data, $file, $filename, $mimes, $real_mime ) {
    $mime_type_values = false;

    $settings = [
      'mime_type_values'          => 'a:1:{i:0;s:25:"gpx = application/gpx+xml";}',
      'security_attempt_enable'   => 'no',
      'filename_sanitized_enable' => 'no'
    ];


    if ( ! isset( $settings['mime_type_values'] ) || empty( $settings['mime_type_values'] ) ) {
      return compact( 'ext', 'type', 'proper_filename' );
    } else {
      $mime_type_values = unserialize( $settings['mime_type_values'] );
    }

    $ext = $type = $proper_filename = false;
    if ( isset( $data['ext'] ) ) {
      $ext = $data['ext'];
    }
    if ( isset( $data['type'] ) ) {
      $ext = $data['type'];
    }
    if ( isset( $data['proper_filename'] ) ) {
      $ext = $data['proper_filename'];
    }
    if ( $ext != false && $type != false ) {
      return $data;
    }

    // If file extension is 2 or more
    $f_sp = explode( ".", $filename );
    $f_exp_count = count( $f_sp );

    // Filename type is "XXX" (There is not file extension).
    if ( $f_exp_count <= 1 ) {
      return $data;
      /* Even if the file extension is "XXX.ZZZ", "XXX.YYY.ZZZ", "AAA.XXX.YYY.ZZZ" or more, it always picks up  the tail of the extensions.
*/
    } else {
      $f_name = $f_sp[0];
      $f_ext = $f_sp[ $f_exp_count - 1 ];
      // WordPress sanitizes the filename in case of 2 or more extensions.
      // ex. XXX.YYY.ZZZ --> XXX_.YYY.ZZZ.
      // The following function fixes the sanitized extension when a file is uploaded in the media in case of allowed extensions.
      // ex. XXX.YYY.ZZZ -- sanitized --> XXX_.YYY.ZZZ -- fixed the plugin --> XXX.YYY.ZZZ
      // In detail, please see sanitize_file_name function in "wp-includes/formatting.php".
      //var_dump($settings['filename_sanitized_enable']);
      if ( isset( $settings['filename_sanitized_enable'] ) && $settings['filename_sanitized_enable'] === "yes" ) {
      } else {
        add_filter( 'sanitize_file_name', function( $filename, $filename_raw ) {
          return str_replace( "_.", ".", $filename );
        }, 10, 2 );
      }
    }

    // If "security_attempt_enable" option disables (default) in the admin menu, the plugin avoids the security check regarding a file extension by WordPress core because of flexible management.
    if ( isset( $settings['security_attempt_enable'] ) && $settings['security_attempt_enable'] === "yes" ) {
      return $data;
    }

    $flag = false;
    if ( ! empty( $mime_type_values ) ) {
      foreach ( (array) $mime_type_values as $line ) {
        // Ignore to the right of '#' on a line.
        $line = substr( $line, 0, strcspn( $line, '#' ) );
        // Escape Strings
        $line = wp_strip_all_tags( $line );

        $line_value = explode( "=", $line );
        if ( count( $line_value ) != 2 ) {
          continue;
        }
        // "　" is the Japanese multi-byte space. If the character is found out, it automatically change the space.
        if ( trim( $line_value[0] ) === $f_ext ) {
          $ext = $f_ext;
          $type = trim( str_replace( "　", " ", $line_value[1] ) );
          $flag = true;
          break;
        }
      }
    }

    if ( $flag ) {
      return compact( 'ext', 'type', 'proper_filename' );
    } else {
      return $data;
    }
  }

  /**
   * Looks at the current screen and loads the correct list table handler.
   *
   * @since 3.3.0
   */
  public function setup_screen() {
    global $twer_list_table;

    $screen_id = false;

    if ( function_exists( 'get_current_screen' ) ) {
      $screen = get_current_screen();
      $screen_id = isset( $screen, $screen->id ) ? $screen->id : '';
    }

    if ( ! empty( $_REQUEST['screen'] ) ) { // WPCS: input var ok.
      $screen_id = twer_clean( wp_unslash( $_REQUEST['screen'] ) ); // WPCS: input var ok, sanitization ok.
    }

    switch ( $screen_id ) {
      case 'edit-map':
        include_once 'list-tables/class-twer-admin-list-table-map.php';
        $twer_list_table = new TWER_Admin_List_Table_Map();
        break;
      case 'edit-marker':
        include_once 'list-tables/class-twer-admin-list-table-marker.php';
        $twer_list_table = new TWER_Admin_List_Table_Marker();
        break;
      case 'edit-route':
        include_once 'list-tables/class-twer-admin-list-table-route.php';
        $twer_list_table = new TWER_Admin_List_Table_Route();
        break;
      case 'edit-twer-custom-fields':
        include_once 'list-tables/class-twer-admin-list-table-custom-fields.php';
        $twer_list_table = new TWER_Admin_List_Table_Custom_Fields();
        break;
      case 'edit-twer-templates':
        include_once 'list-tables/class-twer-admin-list-table-templates.php';
        $twer_list_table = new TWER_Admin_List_Table_Templates();
        break;
      case 'edit-twer-shapes':
        include_once 'list-tables/class-twer-admin-list-table-shapes.php';
        $twer_list_table = new TWER_Admin_List_Table_Shapes();
        break;
    }

    // Ensure the table handler is only loaded once. Prevents multiple loads if a plugin calls check_ajax_referer many times.
    remove_action( 'current_screen', [ $this, 'setup_screen' ] );
    remove_action( 'check_ajax_referer', [ $this, 'setup_screen' ] );
  }

  /**
   * Change messages when a post type is updated.
   *
   * @param array $messages Array of messages.
   *
   * @return array
   */
  public function post_updated_messages( $messages ) {
    global $post;

    $messages['map'] = [
      0  => '', // Unused. Messages start at index 1.
      1  => esc_html__( 'Map updated.', 'treweler' ),
      2  => esc_html__( 'Custom field updated.', 'treweler' ),
      3  => esc_html__( 'Custom field deleted.', 'treweler' ),
      4  => esc_html__( 'Map updated.', 'treweler' ),
      5  => esc_html__( 'Revision restored.', 'treweler' ),
      6  => esc_html__( 'Map published.', 'treweler' ),
      7  => esc_html__( 'Map saved.', 'treweler' ),
      8  => esc_html__( 'Map submitted.', 'treweler' ),
      9  => sprintf(
      /* translators: %s: date */
        '%s %s.',
        esc_html__( 'Map scheduled for:', 'treweler' ),
        '<strong>' . date_i18n( esc_html__( 'M j, Y @ G:i', 'treweler' ), strtotime( $post->post_date ) ) . '</strong>'
      ),
      /* translators: %s: map url */
      10 => esc_html__( 'Map draft updated.', 'treweler' )
    ];

    $messages['route'] = [
      0  => '', // Unused. Messages start at index 1.
      1  => esc_html__( 'Route updated.', 'treweler' ),
      2  => esc_html__( 'Custom field updated.', 'treweler' ),
      3  => esc_html__( 'Custom field deleted.', 'treweler' ),
      4  => esc_html__( 'Route updated.', 'treweler' ),
      5  => esc_html__( 'Revision restored.', 'treweler' ),
      6  => esc_html__( 'Route updated.', 'treweler' ),
      7  => esc_html__( 'Route saved.', 'treweler' ),
      8  => esc_html__( 'Route submitted.', 'treweler' ),
      9  => sprintf(
      /* translators: %s: date */
        '%s %s.',
        esc_html__( 'Route scheduled for:', 'treweler' ),
        '<strong>' . date_i18n( esc_html__( 'M j, Y @ G:i', 'treweler' ), strtotime( $post->post_date ) ) . '</strong>'
      ),
      10 => esc_html__( 'Route draft updated.', 'treweler' )
    ];

    $messages['twer-custom-fields'] = [
      0  => '', // Unused. Messages start at index 1.
      1  => esc_html__( 'Custom field updated.', 'treweler' ),
      2  => esc_html__( 'Custom field updated.', 'treweler' ),
      3  => esc_html__( 'Custom field deleted.', 'treweler' ),
      4  => esc_html__( 'Custom field updated.', 'treweler' ),
      5  => esc_html__( 'Custom field restored.', 'treweler' ),
      6  => esc_html__( 'Custom field updated.', 'treweler' ),
      7  => esc_html__( 'Custom field saved.', 'treweler' ),
      8  => esc_html__( 'Custom field submitted.', 'treweler' ),
      9  => sprintf(
      /* translators: %s: date */
        '%s %s.',
        esc_html__( 'Custom field scheduled for:', 'treweler' ),
        '<strong>' . date_i18n( esc_html__( 'M j, Y @ G:i', 'treweler' ), strtotime( $post->post_date ) ) . '</strong>'
      ),
      10 => esc_html__( 'Custom field draft updated.', 'treweler' )
    ];

    $messages['twer-templates'] = [
      0  => '', // Unused. Messages start at index 1.
      1  => esc_html__( 'Template updated.', 'treweler' ),
      2  => esc_html__( 'Template updated.', 'treweler' ),
      3  => esc_html__( 'Template deleted.', 'treweler' ),
      4  => esc_html__( 'Template updated.', 'treweler' ),
      5  => esc_html__( 'Template restored.', 'treweler' ),
      6  => esc_html__( 'Template updated.', 'treweler' ),
      7  => esc_html__( 'Template saved.', 'treweler' ),
      8  => esc_html__( 'Template submitted.', 'treweler' ),
      9  => sprintf(
      /* translators: %s: date */
        '%s %s.',
        esc_html__( 'Template scheduled for:', 'treweler' ),
        '<strong>' . date_i18n( esc_html__( 'M j, Y @ G:i', 'treweler' ), strtotime( $post->post_date ) ) . '</strong>'
      ),
      10 => esc_html__( 'Template draft updated.', 'treweler' )
    ];

    $messages['twer-shapes'] = [
      0  => '', // Unused. Messages start at index 1.
      1  => esc_html__( 'Shape updated.', 'treweler' ),
      2  => esc_html__( 'Shape updated.', 'treweler' ),
      3  => esc_html__( 'Shape deleted.', 'treweler' ),
      4  => esc_html__( 'Shape updated.', 'treweler' ),
      5  => esc_html__( 'Shape restored.', 'treweler' ),
      6  => esc_html__( 'Shape updated.', 'treweler' ),
      7  => esc_html__( 'Shape saved.', 'treweler' ),
      8  => esc_html__( 'Shape submitted.', 'treweler' ),
      9  => sprintf(
      /* translators: %s: date */
        '%s %s.',
        esc_html__( 'Shape scheduled for:', 'treweler' ),
        '<strong>' . date_i18n( esc_html__( 'M j, Y @ G:i', 'treweler' ), strtotime( $post->post_date ) ) . '</strong>'
      ),
      10 => esc_html__( 'Shape draft updated.', 'treweler' )
    ];

    $messages['marker'] = [
      0  => '', // Unused. Messages start at index 1.
      1  => esc_html__( 'Marker updated.', 'treweler' ),
      2  => esc_html__( 'Custom field updated.', 'treweler' ),
      3  => esc_html__( 'Custom field deleted.', 'treweler' ),
      4  => esc_html__( 'Marker updated.', 'treweler' ),
      5  => esc_html__( 'Revision restored.', 'treweler' ),
      6  => esc_html__( 'Marker updated.', 'treweler' ),
      7  => esc_html__( 'Marker saved.', 'treweler' ),
      8  => esc_html__( 'Marker submitted.', 'treweler' ),
      9  => sprintf(
      /* translators: %s: date */
        '%s %s.',
        esc_html__( 'Marker scheduled for:', 'treweler' ),
        '<strong>' . date_i18n( esc_html__( 'M j, Y @ G:i', 'treweler' ), strtotime( $post->post_date ) ) . '</strong>'
      ),
      10 => esc_html__( 'Marker draft updated.', 'treweler' ),
    ];

    return $messages;
  }

  /**
   * Specify custom bulk actions messages for different post types.
   *
   * @param array $bulk_messages Array of messages.
   * @param array $bulk_counts Array of how many objects were updated.
   *
   * @return array
   */
  public function bulk_post_updated_messages( $bulk_messages, $bulk_counts ) {
    $bulk_messages['map'] = [
      /* translators: %s: map count */
      'updated'   => _n( '%s map updated.', '%s maps updated.', $bulk_counts['updated'], 'treweler' ),
      /* translators: %s: map count */
      'locked'    => _n( '%s map not updated, somebody is editing it.',
        '%s maps not updated, somebody is editing them.', $bulk_counts['locked'], 'treweler' ),
      /* translators: %s: map count */
      'deleted'   => _n( '%s map permanently deleted.', '%s maps permanently deleted.', $bulk_counts['deleted'],
        'treweler' ),
      /* translators: %s: map count */
      'trashed'   => _n( '%s map moved to the Trash.', '%s maps moved to the Trash.', $bulk_counts['trashed'],
        'treweler' ),
      /* translators: %s: map count */
      'untrashed' => _n( '%s map restored from the Trash.', '%s maps restored from the Trash.',
        $bulk_counts['untrashed'], 'treweler' ),
    ];

    $bulk_messages['route'] = [
      /* translators: %s: route count */
      'updated'   => _n( '%s route updated.', '%s routes updated.', $bulk_counts['updated'], 'treweler' ),
      /* translators: %s: route count */
      'locked'    => _n( '%s route not updated, somebody is editing it.',
        '%s routes not updated, somebody is editing them.', $bulk_counts['locked'], 'treweler' ),
      /* translators: %s: route count */
      'deleted'   => _n( '%s route permanently deleted.', '%s routes permanently deleted.',
        $bulk_counts['deleted'], 'treweler' ),
      /* translators: %s: route count */
      'trashed'   => _n( '%s route moved to the Trash.', '%s routes moved to the Trash.', $bulk_counts['trashed'],
        'treweler' ),
      /* translators: %s: route count */
      'untrashed' => _n( '%s route restored from the Trash.', '%s routes restored from the Trash.',
        $bulk_counts['untrashed'], 'treweler' ),
    ];

    $bulk_messages['marker'] = [
      /* translators: %s: marker count */
      'updated'   => _n( '%s marker updated.', '%s markers updated.', $bulk_counts['updated'], 'treweler' ),
      /* translators: %s: marker count */
      'locked'    => _n( '%s marker not updated, somebody is editing it.',
        '%s markers not updated, somebody is editing them.', $bulk_counts['locked'], 'treweler' ),
      /* translators: %s: marker count */
      'deleted'   => _n( '%s marker permanently deleted.', '%s markers permanently deleted.',
        $bulk_counts['deleted'], 'treweler' ),
      /* translators: %s: marker count */
      'trashed'   => _n( '%s marker moved to the Trash.', '%s markers moved to the Trash.',
        $bulk_counts['trashed'], 'treweler' ),
      /* translators: %s: marker count */
      'untrashed' => _n( '%s marker restored from the Trash.', '%s markers restored from the Trash.',
        $bulk_counts['untrashed'], 'treweler' ),
    ];

    $bulk_messages['twer-custom-fields'] = [
      /* translators: %s: custom field count */
      'updated'   => _n( '%s custom field updated.', '%s custom fields updated.', $bulk_counts['updated'], 'treweler' ),
      /* translators: %s: custom field count */
      'locked'    => _n( '%s custom field not updated, somebody is editing it.',
        '%s custom fields not updated, somebody is editing them.', $bulk_counts['locked'], 'treweler' ),
      /* translators: %s: custom field count */
      'deleted'   => _n( '%s custom field permanently deleted.', '%s custom fields permanently deleted.',
        $bulk_counts['deleted'], 'treweler' ),
      /* translators: %s: custom field count */
      'trashed'   => _n( '%s custom field moved to the Trash.', '%s custom fields moved to the Trash.',
        $bulk_counts['trashed'], 'treweler' ),
      /* translators: %s: custom field count */
      'untrashed' => _n( '%s custom field restored from the Trash.', '%s custom fields restored from the Trash.',
        $bulk_counts['untrashed'], 'treweler' ),
    ];

    $bulk_messages['twer-templates'] = [
      /* translators: %s: custom field count */
      'updated'   => _n( '%s template updated.', '%s templates updated.', $bulk_counts['updated'], 'treweler' ),
      /* translators: %s: custom field count */
      'locked'    => _n( '%s template not updated, somebody is editing it.',
        '%s templates not updated, somebody is editing them.', $bulk_counts['locked'], 'treweler' ),
      /* translators: %s: custom field count */
      'deleted'   => _n( '%s template permanently deleted.', '%s templates permanently deleted.',
        $bulk_counts['deleted'], 'treweler' ),
      /* translators: %s: custom field count */
      'trashed'   => _n( '%s template moved to the Trash.', '%s templates moved to the Trash.',
        $bulk_counts['trashed'], 'treweler' ),
      /* translators: %s: custom field count */
      'untrashed' => _n( '%s template restored from the Trash.', '%s templates restored from the Trash.',
        $bulk_counts['untrashed'], 'treweler' ),
    ];

    $bulk_messages['twer-shapes'] = [
      /* translators: %s: custom field count */
      'updated'   => _n( '%s shape updated.', '%s shapes updated.', $bulk_counts['updated'], 'treweler' ),
      /* translators: %s: custom field count */
      'locked'    => _n( '%s shape not updated, somebody is editing it.',
        '%s shapes not updated, somebody is editing them.', $bulk_counts['locked'], 'treweler' ),
      /* translators: %s: custom field count */
      'deleted'   => _n( '%s shape permanently deleted.', '%s shapes permanently deleted.',
        $bulk_counts['deleted'], 'treweler' ),
      /* translators: %s: custom field count */
      'trashed'   => _n( '%s shape moved to the Trash.', '%s shapes moved to the Trash.',
        $bulk_counts['trashed'], 'treweler' ),
      /* translators: %s: custom field count */
      'untrashed' => _n( '%s shape restored from the Trash.', '%s shapes restored from the Trash.',
        $bulk_counts['untrashed'], 'treweler' ),
    ];

    return $bulk_messages;
  }


  /**
   * Output extra data on post forms.
   *
   * @param WP_Post $post Current post object.
   */
  public function edit_form_top( $post ) {
    echo '<input type="hidden" id="original_post_title" name="_treweler_original_post_title" value="' . esc_attr( $post->post_title ) . '" />';
  }

  /**
   * Change title boxes in admin.
   *
   * @param string $text Text to shown.
   * @param WP_Post $post Current post object.
   *
   * @return string
   */
  public function enter_title_here( $text, $post ) {
    switch ( $post->post_type ) {
      case 'map':
        $text = esc_html__( 'Map name', 'treweler' );
        break;
      case 'marker':
        $text = esc_html__( 'Marker name', 'treweler' );
        break;
      case 'route':
        $text = esc_html__( 'Route name', 'treweler' );
        break;
      case 'twer-custom-fields':
        $text = esc_html__( 'Field Name', 'treweler' );
        break;
      case 'twer-templates':
        $text = esc_html__( 'Template Name', 'treweler' );
        break;
      case 'twer-shapes':
        $text = esc_html__( 'Shape Name', 'treweler' );
        break;
    }

    return $text;
  }

  /**
   * Print map/marker/route init fields.
   *
   * @param WP_Post $post Current post object.
   */
  public function edit_form_after_title( $post ) {
    $post_type = isset( $post->post_type ) ? $post->post_type : '';
    if ( twer_is_valid_apikey() ) {
      echo '<div id="js-twer-body" class="twer-body">';
      if ( 'map' === $post_type ) {
        echo '<div id="map" class="treweler-mapbox"></div>';
      } elseif ( 'marker' === $post_type || 'twer-templates' === $post_type ) {
        echo '<div id="marker_map" class="treweler-mapbox"></div>';
      } elseif ( 'route' === $post_type ) {
        $routeProfile = get_post_meta( $post->ID, '_treweler_route_profile', true );
        $routeProfile = ( $routeProfile != '' ? $routeProfile : 'no' );
        ?>
          <div class="mapbox-directions-profile" id="js-twer-directions">
              <input id="mapbox-directions-profile-no" type="radio" name="_treweler_route_profile"
                     value="no" <?php echo ( $routeProfile == 'no' ) ? 'checked' : ''; ?>>
              <label for="mapbox-directions-profile-no"><?php echo esc_html__( 'No Match',
                  'treweler' ); ?></label>
              <input id="mapbox-directions-profile-driving" type="radio" name="_treweler_route_profile"
                     value="driving" <?php echo ( $routeProfile == 'driving' ) ? 'checked' : ''; ?>>
              <label for="mapbox-directions-profile-driving"><?php echo esc_html__( 'Driving',
                  'treweler' ); ?></label>
              <input id="mapbox-directions-profile-walking" type="radio" name="_treweler_route_profile"
                     value="walking" <?php echo ( $routeProfile == 'walking' ) ? 'checked' : ''; ?>>
              <label for="mapbox-directions-profile-walking"><?php echo esc_html__( 'Walking',
                  'treweler' ); ?></label>
              <input id="mapbox-directions-profile-cycling" type="radio" name="_treweler_route_profile"
                     value="cycling" <?php echo ( $routeProfile == 'cycling' ) ? 'checked' : ''; ?>>
              <label for="mapbox-directions-profile-cycling"><?php echo esc_html__( 'Cycling',
                  'treweler' ); ?></label>
          </div>
          <div id="js-twer-route-map" class="treweler-mapbox"></div>
          <div class="info-box">
              <div id="info">
                  <p><?php echo esc_html__( 'Draw your route using the draw tools in the upper right corner of the map. To get the most accurate route match, draw points at regular intervals.',
                      'treweler' ); ?></p>
              </div>
              <div id="directions"></div>
          </div>
        <?php
      } elseif ( 'twer-shapes' === $post_type ) {
        $message_text = '';
        ob_start();
        ?>
          <p><?php printf( esc_html__( 'In draw mode :%s
    - Alt + Drag with left click to create nodes with bezier handles.%s
    - Left click to create nodes without handles.', 'treweler' ), '<br>', '<br>' ); ?></p>

          <p><?php printf( esc_html__( 'In edit mode :%s
              - Drag with Left click on a handle to move it.%s
              - Alt + Drag with left click on a handle to break handle symetry.', 'treweler' ), '<br>', '<br>' ); ?></p>
        <?php
        $message_text = ob_get_clean();

        echo '<div id="js-twer-map-msg" class="twer-map-message">' . $message_text . '</div>';
        echo '<div id="js-twer-map" class="twer-map"></div>';
      }

      echo '</div>';
    }
  }

  /**
   * Hidden default Meta-Boxes.
   *
   * @param array $hidden Hidden boxes.
   * @param object $screen Current screen.
   *
   * @return array
   */
  public function hidden_meta_boxes( $hidden, $screen ) {
    if ( 'map' === $screen->post_type && 'post' === $screen->base ) {
      $hidden = array_merge( $hidden, [ 'postcustom' ] );
    }

    return $hidden;
  }

}

new TWER_Admin_Post_Types();
